terraform {
  required_providers {
    aws = {
      source  = "hashicorp/aws"
      version = "~> 5.0"
    }
  }
}

data "aws_region" "current" {}

data "aws_caller_identity" "current" {}

variable "confidence-user-name" {
  default = "confidence-user"
}

variable "confidence-role-name" {
  default = "confidence-role"
}

variable "confidence-policy-name" {
  default = "confidence-policy"
}

variable "redshift-cluster-name" {}

output "redshift-cluster-name" {
  description = "The Redshift cluster"
  value       = var.redshift-cluster-name
}

output "access_key_id" {
  description = "The access key id for the IAM user"
  value       = aws_iam_access_key.confidence-user-access-key.id
}

output "access_key_secret" {
  description = "The access key secret for the IAM user"
  sensitive   = true
  value       = aws_iam_access_key.confidence-user-access-key.secret
}

output "role_arn" {
  description = "The role ARN to configure in the confidence settings"
  value       = aws_iam_role.confidence-role.arn
}

output "redshift-sql" {
  description = "The SQL required to give the new user permissions in redshift"
  value       = <<EOT
CREATE database confidence;
CREATE SCHEMA confidence;
CREATE USER "IAMR:${var.confidence-role-name}" PASSWORD DISABLE;
GRANT ALL ON SCHEMA confidence TO "IAMR:${var.confidence-role-name}";
GRANT ALL ON ALL TABLES IN SCHEMA confidence TO "IAMR:${var.confidence-role-name}";
ALTER DEFAULT PRIVILEGES FOR USER "IAMR:${var.confidence-role-name}" IN SCHEMA confidence GRANT ALL ON TABLES TO "IAMR:${var.confidence-role-name}";

EOT
}


provider "aws" {
}

resource "aws_iam_user" "confidence-user" {
  name = var.confidence-user-name
  tags = {
    type = "service"
  }
}

resource "aws_iam_access_key" "confidence-user-access-key" {
  user = aws_iam_user.confidence-user.name
}

resource "aws_s3_bucket" "confidence-s3-bucket" {
  bucket_prefix = "confidence-"
}

resource "aws_s3_bucket_public_access_block" "bucket-public-access-block" {
  bucket = aws_s3_bucket.confidence-s3-bucket.id

  block_public_acls       = true
  block_public_policy     = true
  ignore_public_acls      = true
  restrict_public_buckets = true
}

resource "aws_iam_role" "confidence-role" {
  name = var.confidence-role-name

  assume_role_policy = jsonencode({
    Version = "2012-10-17"
    Statement = [
      {
        Action = "sts:AssumeRole"
        Effect = "Allow"
        Sid    = ""
        Principal = {
          Service = ["redshift.amazonaws.com", "redshift-serverless.amazonaws.com"]
        }
      },
      {
        Action = "sts:AssumeRole"
        Effect = "Allow"
        Sid    = ""
        Principal = {
          AWS = aws_iam_user.confidence-user.arn
        }
      }
    ]
  })

  inline_policy {
    name   = var.confidence-policy-name
    policy = data.aws_iam_policy_document.confidence-policy.json
  }
}

data "aws_iam_policy_document" "confidence-policy" {
  statement {
    actions = [
      "s3:ListBucket"
    ]
    resources = ["${aws_s3_bucket.confidence-s3-bucket.arn}"]
    effect    = "Allow"
  }
  statement {
    actions = [
      "s3:GetObject",
      "s3:GetObjectAcl",
      "s3:PutObject",
      "s3:PutObjectAcl"
    ]
    resources = ["${aws_s3_bucket.confidence-s3-bucket.arn}/*"]
    effect    = "Allow"
  }
  statement {
    actions = [
      "redshift-data:DescribeStatement",
      "redshift-data:GetStatementResult",
      "redshift-data:CancelStatement",
      "redshift-data:ListStatements"
    ]
    resources = ["*"]
    effect    = "Allow"
  }
  statement {
    actions = [
      "redshift:GetClusterCredentials",
			"redshift:GetClusterCredentialsWithIAM",
      "redshift-data:ListTables",
      "redshift-data:DescribeTable",
      "redshift-data:ListDatabases",
      "redshift-data:ListSchemas",
      "redshift-data:BatchExecuteStatement",
      "redshift-data:ExecuteStatement"
    ]
    resources = [
      "arn:aws:redshift:${data.aws_region.current.name}:${data.aws_caller_identity.current.account_id}:cluster:${var.redshift-cluster-name}",
      "arn:aws:redshift:${data.aws_region.current.name}:${data.aws_caller_identity.current.account_id}:dbname:${var.redshift-cluster-name}/*"
    ]
    effect = "Allow"
  }
}
