variable "account-login" {
  type = string
  default = "company"
  description = "The id that you use to login to your organization."
}

variable "gcp-project" {
  type = string
  default = ""
  description = "The GCP project that you want to create the resources in."
}

variable "gcp-region" {
  type = string
  default = ""
  description = "The region that you want to create GCP resources in."
}

variable "service-account-name" {
  type = string
  default = "confidence"
  description = "Name of the Confidence service account."
}

variable "bq-project" {
  type = string
  default = ""
  description = "Name of the BigQuery project that you want to create datasets in, if empty uses gcp-project instead."
}

variable "bq-assignment-dataset" {
  type = string
  default = "confidence_assignments"
  description = "Id of the BigQuery dataset to store assignments in."
}

variable "bq-exposure-dataset" {
  type = string
  default = "confidence_exposures"
  description = "Id of the BigQuery dataset to store exposure tables in."
}

variable "bq-location" {
  type = string
  default = "EU"
  description = "Location you want the BigQuery datasets to be created in."
}

provider "google" {
  project     = var.gcp-project
  region      = var.gcp-region
}

resource "google_service_account" "confidence_service_account" {
  account_id   = var.service-account-name
  display_name = "Confidence Service Account"
}

resource "google_project_iam_member" "confidence_iam_member" {
  project = var.gcp-project
  role    = "roles/bigquery.jobUser"
  member  = "serviceAccount:${google_service_account.confidence_service_account.email}"
}

resource "google_service_account_iam_member" "confidence_impersonate_iam_binding" {
  service_account_id = "projects/${var.gcp-project}/serviceAccounts/${google_service_account.confidence_service_account.email}"
  role               = "roles/iam.workloadIdentityUser"
  member             = "serviceAccount:account-${var.account-login}@spotify-confidence.iam.gserviceaccount.com"
}

resource "google_bigquery_dataset" "confidence_assignments_dataset" {
  project   = var.bq-project != "" ? var.bq-project : var.gcp-project
  dataset_id = var.bq-assignment-dataset
  location  = var.bq-location
  access {
    role = "OWNER"
    user_by_email = google_service_account.confidence_service_account.email
  }
}

resource "google_bigquery_dataset" "confidence_exposures_dataset" {
  project   = var.bq-project != "" ? var.bq-project : var.gcp-project
  dataset_id = var.bq-exposure-dataset
  location  = var.bq-location
  access {
    role = "OWNER"
    user_by_email = google_service_account.confidence_service_account.email
  }
}
